// Global variables
let currentUser = null;
let authToken = null;
let employees = [];
let flags = [];
let users = [];

// API Base URL
const API_BASE = '/api';

// DOM Elements
const loginScreen = document.getElementById('loginScreen');
const mainApp = document.getElementById('mainApp');
const loading = document.getElementById('loading');

// Initialize the application
document.addEventListener('DOMContentLoaded', function() {
    initializeTheme();
    setupEventListeners();
    checkAuthStatus();
});

// Theme Management
function initializeTheme() {
    const savedTheme = localStorage.getItem('theme') || 'light';
    if (savedTheme === 'dark') {
        document.documentElement.classList.add('dark');
    }
    
    // Theme toggle buttons
    document.getElementById('themeToggle').addEventListener('click', toggleTheme);
    document.getElementById('themeToggleMain').addEventListener('click', toggleTheme);
}

function toggleTheme() {
    document.documentElement.classList.toggle('dark');
    const isDark = document.documentElement.classList.contains('dark');
    localStorage.setItem('theme', isDark ? 'dark' : 'light');
}

// Event Listeners Setup
function setupEventListeners() {
    // Login form
    document.getElementById('loginForm').addEventListener('submit', handleLogin);
    
    // Logout button
    document.getElementById('logoutBtn').addEventListener('click', handleLogout);
    
    // Navigation tabs
    document.getElementById('dashboardTab').addEventListener('click', () => showSection('dashboard'));
    document.getElementById('flagsTab').addEventListener('click', () => showSection('flags'));
    document.getElementById('employeesTab').addEventListener('click', () => showSection('employees'));
    document.getElementById('usersTab').addEventListener('click', () => showSection('users'));
    document.getElementById('auditTab').addEventListener('click', () => showSection('audit'));
    
    // Add buttons
    document.getElementById('addFlagBtn').addEventListener('click', () => openFlagModal());
    document.getElementById('addEmployeeBtn').addEventListener('click', () => openEmployeeModal());
    document.getElementById('addUserBtn').addEventListener('click', () => openUserModal());
    
    // Filter buttons
    document.getElementById('applyFiltersBtn').addEventListener('click', applyFilters);
    document.getElementById('clearFiltersBtn').addEventListener('click', clearFilters);
    
    // Modal forms
    document.getElementById('flagForm').addEventListener('submit', handleFlagSubmit);
    document.getElementById('employeeForm').addEventListener('submit', handleEmployeeSubmit);
    document.getElementById('userForm').addEventListener('submit', handleUserSubmit);
    
    // Modal cancel buttons
    document.getElementById('cancelFlagBtn').addEventListener('click', () => closeModal('flagModal'));
    document.getElementById('cancelEmployeeBtn').addEventListener('click', () => closeModal('employeeModal'));
    document.getElementById('cancelUserBtn').addEventListener('click', () => closeModal('userModal'));
}

// Authentication
function checkAuthStatus() {
    const token = localStorage.getItem('authToken');
    const user = localStorage.getItem('currentUser');
    
    if (token && user) {
        authToken = token;
        currentUser = JSON.parse(user);
        showMainApp();
    } else {
        showLoginScreen();
    }
}

async function handleLogin(e) {
    e.preventDefault();
    
    const username = document.getElementById('username').value;
    const password = document.getElementById('password').value;
    
    showLoading(true);
    
    try {
        const response = await fetch(`${API_BASE}/auth/login`, {
            method: 'POST',
            headers: {
                'Content-Type': 'application/json'
            },
            body: JSON.stringify({ username, password })
        });
        
        const data = await response.json();
        
        if (response.ok) {
            authToken = data.token;
            currentUser = data.user;
            
            localStorage.setItem('authToken', authToken);
            localStorage.setItem('currentUser', JSON.stringify(currentUser));
            
            showMainApp();
        } else {
            showError(data.error || 'Erro no login');
        }
    } catch (error) {
        console.error('Erro no login:', error);
        showError('Erro de conexão');
    } finally {
        showLoading(false);
    }
}

function handleLogout() {
    localStorage.removeItem('authToken');
    localStorage.removeItem('currentUser');
    authToken = null;
    currentUser = null;
    showLoginScreen();
}

// UI Management
function showLoginScreen() {
    loginScreen.classList.remove('hidden');
    mainApp.classList.add('hidden');
}

function showMainApp() {
    loginScreen.classList.add('hidden');
    mainApp.classList.remove('hidden');
    
    // Update user info
    document.getElementById('userInfo').textContent = `${currentUser.username} (${currentUser.role})`;
    
    // Show/hide admin sections
    if (currentUser.role === 'Admin') {
        document.getElementById('usersTab').classList.remove('hidden');
        document.getElementById('auditTab').classList.remove('hidden');
    } else {
        document.getElementById('usersTab').classList.add('hidden');
        document.getElementById('auditTab').classList.add('hidden');
    }
    
    // Load initial data
    showSection('dashboard');
}

function showSection(section) {
    // Hide all sections
    document.querySelectorAll('.section').forEach(s => s.classList.add('hidden'));
    
    // Remove active class from all tabs
    document.querySelectorAll('.tab').forEach(t => t.classList.remove('active'));
    
    // Show selected section
    document.getElementById(`${section}Section`).classList.remove('hidden');
    document.getElementById(`${section}Tab`).classList.add('active');
    
    // Load section data
    switch(section) {
        case 'dashboard':
            loadDashboard();
            break;
        case 'flags':
            loadFlags();
            break;
        case 'employees':
            loadEmployees();
            break;
        case 'users':
            loadUsers();
            break;
        case 'audit':
            loadAuditLogs();
            break;
    }
}

function showLoading(show) {
    if (show) {
        loading.classList.remove('hidden');
    } else {
        loading.classList.add('hidden');
    }
}

function showError(message) {
    // Create error notification
    const errorDiv = document.createElement('div');
    errorDiv.className = 'fixed top-4 right-4 bg-red-500 text-white px-4 py-2 rounded shadow-lg z-50';
    errorDiv.textContent = message;
    
    document.body.appendChild(errorDiv);
    
    setTimeout(() => {
        document.body.removeChild(errorDiv);
    }, 3000);
}

function showSuccess(message) {
    // Create success notification
    const successDiv = document.createElement('div');
    successDiv.className = 'fixed top-4 right-4 bg-green-500 text-white px-4 py-2 rounded shadow-lg z-50';
    successDiv.textContent = message;
    
    document.body.appendChild(successDiv);
    
    setTimeout(() => {
        document.body.removeChild(successDiv);
    }, 3000);
}

// Dashboard Functions
async function loadDashboard() {
    try {
        showLoading(true);
        
        // Load flags and employees for dashboard
        await Promise.all([
            loadFlags(true), // Load for dashboard
            loadEmployees(true) // Load for dashboard
        ]);
        
        updateDashboardStats();
        updateRecentFlags();
        
    } catch (error) {
        console.error('Erro ao carregar dashboard:', error);
        showError('Erro ao carregar dashboard');
    } finally {
        showLoading(false);
    }
}

function updateDashboardStats() {
    const totalFlags = flags.length;
    const closedFlags = flags.filter(f => f.status === 'Concluída').length;
    const openFlags = flags.filter(f => ['Aberta', 'Em análise', 'Em plano de ação'].includes(f.status)).length;
    const totalEmployees = employees.length;
    
    document.getElementById('totalFlags').textContent = totalFlags;
    document.getElementById('closedFlags').textContent = closedFlags;
    document.getElementById('openFlags').textContent = openFlags;
    document.getElementById('totalEmployees').textContent = totalEmployees;
}

function updateRecentFlags() {
    const recentFlags = flags.slice(0, 5); // Show last 5 flags
    const container = document.getElementById('recentFlagsList');
    
    // If container doesn't exist, skip this function
    if (!container) {
        return;
    }
    
    if (recentFlags.length === 0) {
        container.innerHTML = '<p class="text-gray-500 dark:text-gray-400">Nenhuma flag encontrada</p>';
        return;
    }
    
    container.innerHTML = recentFlags.map(flag => `
        <div class="bg-white dark:bg-gray-800 p-4 rounded-lg shadow cursor-pointer hover:shadow-md transition-shadow"
             onclick="openFlagDetails(${flag.id})">
            <div class="flex justify-between items-start mb-2">
                <h4 class="font-medium text-gray-900 dark:text-white">${flag.title}</h4>
                <span class="px-2 py-1 text-xs rounded-full ${getGravityColor(flag.gravity)}">${flag.gravity}</span>
            </div>
            <p class="text-sm text-gray-600 dark:text-gray-300 mb-2">${flag.employee_name}</p>
            <div class="flex justify-between items-center">
                <span class="text-xs text-gray-500">${formatDate(flag.created_at)}</span>
                <span class="px-2 py-1 text-xs rounded-full ${getStatusColor(flag.status)}">${flag.status}</span>
            </div>
        </div>
    `).join('');
}

// Flags Functions
async function loadFlags(forDashboard = false) {
    try {
        const params = new URLSearchParams();
        
        // For dashboard, don't include archived flags by default
        if (forDashboard) {
            params.append('include_archived', 'false');
        }
        
        const response = await fetch(`${API_BASE}/flags?${params}`, {
            headers: {
                'Authorization': `Bearer ${authToken}`
            }
        });
        
        if (response.ok) {
            flags = await response.json();
            
            if (!forDashboard) {
                updateFlagsTable();
                updateFlagFilters();
            }
        } else {
            throw new Error('Erro ao carregar flags');
        }
    } catch (error) {
        console.error('Erro ao carregar flags:', error);
        if (!forDashboard) {
            showError('Erro ao carregar flags');
        }
    }
}

function updateFlagsTable() {
    const tbody = document.getElementById('flagsTableBody');
    
    if (flags.length === 0) {
        tbody.innerHTML = '<tr><td colspan="6" class="text-center py-4 text-gray-500">Nenhuma flag encontrada</td></tr>';
        return;
    }
    
    tbody.innerHTML = flags.map(flag => `
        <tr class="hover:bg-gray-50 dark:hover:bg-gray-700 cursor-pointer" onclick="openFlagDetails(${flag.id})">
            <td class="px-6 py-4 whitespace-nowrap text-sm text-gray-900 dark:text-white">
                ${formatDate(flag.created_at)}
            </td>
            <td class="px-6 py-4 whitespace-nowrap">
                <span class="px-2 py-1 text-xs rounded-full ${getGravityColor(flag.gravity)}">${flag.gravity}</span>
            </td>
            <td class="px-6 py-4 whitespace-nowrap text-sm text-gray-900 dark:text-white">
                ${flag.employee_name}
            </td>
            <td class="px-6 py-4 whitespace-nowrap text-sm text-gray-900 dark:text-white">
                ${flag.leader_name}
            </td>
            <td class="px-6 py-4 whitespace-nowrap">
                <select class="text-xs rounded-full px-2 py-1 ${getStatusColor(flag.status)}" 
                        onchange="updateFlagStatus(${flag.id}, this.value)" onclick="event.stopPropagation()">
                    <option value="Aberta" ${flag.status === 'Aberta' ? 'selected' : ''}>Aberta</option>
                    <option value="Em análise" ${flag.status === 'Em análise' ? 'selected' : ''}>Em análise</option>
                    <option value="Em plano de ação" ${flag.status === 'Em plano de ação' ? 'selected' : ''}>Em plano de ação</option>
                    <option value="Concluída" ${flag.status === 'Concluída' ? 'selected' : ''}>Concluída</option>
                    <option value="Arquivada" ${flag.status === 'Arquivada' ? 'selected' : ''}>Arquivada</option>
                </select>
            </td>
            <td class="px-6 py-4 whitespace-nowrap text-sm text-gray-900 dark:text-white">
                ${flag.title}
            </td>
        </tr>
    `).join('');
}

async function updateFlagStatus(flagId, newStatus) {
    try {
        const response = await fetch(`${API_BASE}/flags/${flagId}/status`, {
            method: 'PATCH',
            headers: {
                'Content-Type': 'application/json',
                'Authorization': `Bearer ${authToken}`
            },
            body: JSON.stringify({ status: newStatus })
        });
        
        if (response.ok) {
            showSuccess('Status atualizado com sucesso');
            // Update local flag data
            const flagIndex = flags.findIndex(f => f.id === flagId);
            if (flagIndex !== -1) {
                flags[flagIndex].status = newStatus;
            }
            // Reload dashboard if we're on it
            if (!document.getElementById('dashboardSection').classList.contains('hidden')) {
                updateDashboardStats();
            }
        } else {
            const data = await response.json();
            showError(data.error || 'Erro ao atualizar status');
        }
    } catch (error) {
        console.error('Erro ao atualizar status:', error);
        showError('Erro ao atualizar status');
    }
}

function updateFlagFilters() {
    // Update employee filter
    const employeeFilter = document.getElementById('filterEmployee');
    employeeFilter.innerHTML = '<option value="">Todos os colaboradores</option>' +
        employees.map(emp => `<option value="${emp.id}">${emp.name}</option>`).join('');
    
    // Update leader filter (only for admins)
    if (currentUser.role === 'Admin') {
        const leaderFilter = document.getElementById('filterLeader');
        if (leaderFilter) {
            leaderFilter.innerHTML = '<option value="">Todos os líderes</option>' +
                users.filter(u => u.role === 'Lider').map(user => `<option value="${user.id}">${user.username}</option>`).join('');
        }
    }
}

// Employees Functions
async function loadEmployees(forDashboard = false) {
    try {
        const response = await fetch(`${API_BASE}/employees`, {
            headers: {
                'Authorization': `Bearer ${authToken}`
            }
        });
        
        if (response.ok) {
            employees = await response.json();
            
            if (!forDashboard) {
                updateEmployeesTable();
            }
        } else {
            throw new Error('Erro ao carregar colaboradores');
        }
    } catch (error) {
        console.error('Erro ao carregar colaboradores:', error);
        if (!forDashboard) {
            showError('Erro ao carregar colaboradores');
        }
    }
}

function updateEmployeesTable() {
    const tbody = document.getElementById('employeesTableBody');
    
    if (employees.length === 0) {
        tbody.innerHTML = '<tr><td colspan="4" class="text-center py-4 text-gray-500">Nenhum colaborador encontrado</td></tr>';
        return;
    }
    
    tbody.innerHTML = employees.map(employee => `
        <tr class="hover:bg-gray-50 dark:hover:bg-gray-700">
            <td class="px-6 py-4 whitespace-nowrap text-sm text-gray-900 dark:text-white">${employee.name}</td>
            <td class="px-6 py-4 whitespace-nowrap text-sm text-gray-900 dark:text-white">${employee.employee_function || '-'}</td>
            <td class="px-6 py-4 whitespace-nowrap text-sm text-gray-900 dark:text-white">${employee.leader_name}</td>
            <td class="px-6 py-4 whitespace-nowrap text-sm text-gray-900 dark:text-white">${formatDate(employee.created_at)}</td>
            <td class="px-6 py-4 whitespace-nowrap text-sm font-medium">
                <button onclick="editEmployee(${employee.id})" class="text-indigo-600 hover:text-indigo-900 mr-2">Editar</button>
                <button onclick="deleteEmployee(${employee.id})" class="text-red-600 hover:text-red-900">Excluir</button>
            </td>
        </tr>
    `).join('');
}

// Users Functions (Admin only)
async function loadUsers() {
    if (currentUser.role !== 'Admin') return;
    
    try {
        const response = await fetch(`${API_BASE}/users`, {
            headers: {
                'Authorization': `Bearer ${authToken}`
            }
        });
        
        if (response.ok) {
            users = await response.json();
            updateUsersTable();
        } else {
            throw new Error('Erro ao carregar usuários');
        }
    } catch (error) {
        console.error('Erro ao carregar usuários:', error);
        showError('Erro ao carregar usuários');
    }
}

function updateUsersTable() {
    const tbody = document.getElementById('usersTableBody');
    
    if (users.length === 0) {
        tbody.innerHTML = '<tr><td colspan="4" class="text-center py-4 text-gray-500">Nenhum usuário encontrado</td></tr>';
        return;
    }
    
    tbody.innerHTML = users.map(user => `
        <tr class="hover:bg-gray-50 dark:hover:bg-gray-700">
            <td class="px-6 py-4 whitespace-nowrap text-sm text-gray-900 dark:text-white">${user.username}</td>
            <td class="px-6 py-4 whitespace-nowrap">
                <span class="px-2 py-1 text-xs rounded-full ${user.role === 'Admin' ? 'bg-purple-100 text-purple-800' : 'bg-blue-100 text-blue-800'}">${user.role}</span>
            </td>
            <td class="px-6 py-4 whitespace-nowrap text-sm text-gray-900 dark:text-white">${formatDate(user.created_at)}</td>
            <td class="px-6 py-4 whitespace-nowrap text-sm font-medium">
                <button onclick="editUser(${user.id})" class="text-indigo-600 hover:text-indigo-900 mr-2">Editar</button>
                <button onclick="deleteUser(${user.id})" class="text-red-600 hover:text-red-900">Excluir</button>
            </td>
        </tr>
    `).join('');
}

// Modal Functions
function openFlagModal(flagId = null) {
    const modal = document.getElementById('flagModal');
    const form = document.getElementById('flagForm');
    const title = document.getElementById('flagModalTitle');
    
    // Reset form
    form.reset();
    
    if (flagId) {
        title.textContent = 'Editar Flag';
        // Load flag data for editing
        const flag = flags.find(f => f.id === flagId);
        if (flag) {
            document.getElementById('flagTitle').value = flag.title;
            document.getElementById('flagDescription').value = flag.description;
            document.getElementById('flagEmployee').value = flag.employee_id;
            document.getElementById('flagCategory').value = flag.category;
            document.getElementById('flagOccurredAt').value = flag.occurred_at;
            document.getElementById('flagGravity').value = flag.gravity;
            document.getElementById('flagImpactArea').value = flag.impact_area;
            document.getElementById('flagImpactOther').value = flag.impact_other;
            document.getElementById('flagStatus').value = flag.status;
        }
        form.dataset.flagId = flagId;
    } else {
        title.textContent = 'Cadastrar Flag';
        delete form.dataset.flagId;
    }
    
    // Populate employee select
    const employeeSelect = document.getElementById('flagEmployee');
    employeeSelect.innerHTML = '<option value="">Selecione um colaborador</option>' +
        employees.map(emp => `<option value="${emp.id}">${emp.name}</option>`).join('');
    
    modal.classList.remove('hidden');
}

function openEmployeeModal(employeeId = null) {
    const modal = document.getElementById('employeeModal');
    const form = document.getElementById('employeeForm');
    const title = document.getElementById('employeeModalTitle');
    
    // Reset form
    form.reset();
    
    if (employeeId) {
        title.textContent = 'Editar Colaborador';
        // Load employee data for editing
        const employee = employees.find(e => e.id === employeeId);
        if (employee) {
            document.getElementById('employeeName').value = employee.name;
            document.getElementById('employeeFunction').value = employee.employee_function;
        }
        form.dataset.employeeId = employeeId;
    } else {
        title.textContent = 'Cadastrar Colaborador';
        delete form.dataset.employeeId;
    }
    
    modal.classList.remove('hidden');
}

function openUserModal(userId = null) {
    if (currentUser.role !== 'Admin') return;
    
    const modal = document.getElementById('userModal');
    const form = document.getElementById('userForm');
    const title = document.getElementById('userModalTitle');
    
    // Reset form
    form.reset();
    
    if (userId) {
        title.textContent = 'Editar Usuário';
        // Load user data for editing
        const user = users.find(u => u.id === userId);
        if (user) {
            document.getElementById('userUsername').value = user.username;
            document.getElementById('userRole').value = user.role;
        }
        form.dataset.userId = userId;
    } else {
        title.textContent = 'Cadastrar Usuário';
        delete form.dataset.userId;
    }
    
    modal.classList.remove('hidden');
}

function openFlagDetails(flagId) {
    // Open flag modal for editing
    openFlagModal(flagId);
}

function closeModal(modalId) {
    document.getElementById(modalId).classList.add('hidden');
}

// Form Handlers
async function handleFlagSubmit(e) {
    e.preventDefault();
    
    const form = e.target;
    const flagId = form.dataset.flagId;
    const isEdit = !!flagId;
    
    const flagData = {
        title: document.getElementById('flagTitle').value,
        description: document.getElementById('flagDescription').value,
        employee_id: parseInt(document.getElementById('flagEmployee').value),
        category: document.getElementById('flagCategory').value,
        occurred_at: document.getElementById('flagOccurredAt').value,
        gravity: document.getElementById('flagGravity').value,
        impact_area: document.getElementById('flagImpactArea').value,
        impact_other: document.getElementById('flagImpactOther').value,
        status: document.getElementById('flagStatus').value
    };
    

    
    console.log('Flag data to submit:', flagData);
    
    // Validação no front-end
    if (!flagData.title || !flagData.employee_id || !flagData.category || !flagData.occurred_at || !flagData.gravity) {
        showError('Por favor, preencha todos os campos obrigatórios');
        return;
    }
    
    try {
        const url = isEdit ? `${API_BASE}/flags/${flagId}` : `${API_BASE}/flags`;
        const method = isEdit ? 'PUT' : 'POST';
        
        console.log('Sending request to:', url, 'with method:', method);
        
        const response = await fetch(url, {
            method,
            headers: {
                'Content-Type': 'application/json',
                'Authorization': `Bearer ${authToken}`
            },
            body: JSON.stringify(flagData)
        });
        
        console.log('Response status:', response.status);
        
        if (response.ok) {
            const result = await response.json();
            console.log('Success response:', result);
            showSuccess(isEdit ? 'Flag atualizada com sucesso' : 'Flag criada com sucesso');
            closeModal('flagModal');
            loadFlags();
            if (!document.getElementById('dashboardSection').classList.contains('hidden')) {
                loadDashboard();
            }
        } else {
            const data = await response.json();
            console.error('Error response:', data);
            showError(data.error || 'Erro ao salvar flag');
        }
    } catch (error) {
        console.error('Erro ao salvar flag:', error);
        showError('Erro ao salvar flag');
    }
}

async function handleEmployeeSubmit(e) {
    e.preventDefault();
    
    const form = e.target;
    const employeeId = form.dataset.employeeId;
    const isEdit = !!employeeId;
    
    const employeeData = {
        name: document.getElementById('employeeName').value,
        employee_function: document.getElementById('employeeFunction').value
    };
    
    try {
        const url = isEdit ? `${API_BASE}/employees/${employeeId}` : `${API_BASE}/employees`;
        const method = isEdit ? 'PUT' : 'POST';
        
        const response = await fetch(url, {
            method,
            headers: {
                'Content-Type': 'application/json',
                'Authorization': `Bearer ${authToken}`
            },
            body: JSON.stringify(employeeData)
        });
        
        if (response.ok) {
            showSuccess(isEdit ? 'Colaborador atualizado com sucesso' : 'Colaborador criado com sucesso');
            closeModal('employeeModal');
            loadEmployees();
            if (!document.getElementById('dashboardSection').classList.contains('hidden')) {
                loadDashboard();
            }
        } else {
            const data = await response.json();
            showError(data.error || 'Erro ao salvar colaborador');
        }
    } catch (error) {
        console.error('Erro ao salvar colaborador:', error);
        showError('Erro ao salvar colaborador');
    }
}

async function handleUserSubmit(e) {
    e.preventDefault();
    
    if (currentUser.role !== 'Admin') return;
    
    const form = e.target;
    const userId = form.dataset.userId;
    const isEdit = !!userId;
    
    const userData = {
        username: document.getElementById('userUsername').value,
        role: document.getElementById('userRole').value
    };
    
    const password = document.getElementById('userPassword').value;
    if (password) {
        userData.password = password;
    }
    
    try {
        const url = isEdit ? `${API_BASE}/users/${userId}` : `${API_BASE}/users`;
        const method = isEdit ? 'PUT' : 'POST';
        
        const response = await fetch(url, {
            method,
            headers: {
                'Content-Type': 'application/json',
                'Authorization': `Bearer ${authToken}`
            },
            body: JSON.stringify(userData)
        });
        
        if (response.ok) {
            showSuccess(isEdit ? 'Usuário atualizado com sucesso' : 'Usuário criado com sucesso');
            closeModal('userModal');
            loadUsers();
        } else {
            const data = await response.json();
            showError(data.error || 'Erro ao salvar usuário');
        }
    } catch (error) {
        console.error('Erro ao salvar usuário:', error);
        showError('Erro ao salvar usuário');
    }
}

// Edit/Delete Functions
function editEmployee(employeeId) {
    openEmployeeModal(employeeId);
}

async function deleteEmployee(employeeId) {
    if (!confirm('Tem certeza que deseja excluir este colaborador?')) return;
    
    try {
        const response = await fetch(`${API_BASE}/employees/${employeeId}`, {
            method: 'DELETE',
            headers: {
                'Authorization': `Bearer ${authToken}`
            }
        });
        
        if (response.ok) {
            showSuccess('Colaborador excluído com sucesso');
            loadEmployees();
            if (!document.getElementById('dashboardSection').classList.contains('hidden')) {
                loadDashboard();
            }
        } else {
            const data = await response.json();
            showError(data.error || 'Erro ao excluir colaborador');
        }
    } catch (error) {
        console.error('Erro ao excluir colaborador:', error);
        showError('Erro ao excluir colaborador');
    }
}

function editUser(userId) {
    openUserModal(userId);
}

async function deleteUser(userId) {
    if (!confirm('Tem certeza que deseja excluir este usuário?')) return;
    
    try {
        const response = await fetch(`${API_BASE}/users/${userId}`, {
            method: 'DELETE',
            headers: {
                'Authorization': `Bearer ${authToken}`
            }
        });
        
        if (response.ok) {
            showSuccess('Usuário excluído com sucesso');
            loadUsers();
        } else {
            const data = await response.json();
            showError(data.error || 'Erro ao excluir usuário');
        }
    } catch (error) {
        console.error('Erro ao excluir usuário:', error);
        showError('Erro ao excluir usuário');
    }
}

// Filter Functions
async function applyFilters() {
    const params = new URLSearchParams();
    
    const employeeFilter = document.getElementById('filterEmployee').value;
    const statusFilter = document.getElementById('filterStatus').value;
    const startDate = document.getElementById('filterStartDate').value;
    const endDate = document.getElementById('filterEndDate').value;
    const includeArchived = document.getElementById('filterIncludeArchived').checked;
    
    if (employeeFilter) params.append('employee_id', employeeFilter);
    if (statusFilter) params.append('status', statusFilter);
    if (startDate) params.append('start_date', startDate);
    if (endDate) params.append('end_date', endDate);
    if (includeArchived) params.append('include_archived', 'true');
    
    // Add leader filter for admins
    if (currentUser.role === 'Admin') {
        const leaderFilter = document.getElementById('filterLeader').value;
        if (leaderFilter) params.append('leader_id', leaderFilter);
    }
    
    try {
        const response = await fetch(`${API_BASE}/flags?${params}`, {
            headers: {
                'Authorization': `Bearer ${authToken}`
            }
        });
        
        if (response.ok) {
            flags = await response.json();
            updateFlagsTable();
        } else {
            throw new Error('Erro ao aplicar filtros');
        }
    } catch (error) {
        console.error('Erro ao aplicar filtros:', error);
        showError('Erro ao aplicar filtros');
    }
}

function clearFilters() {
    document.getElementById('filterEmployee').value = '';
    document.getElementById('filterStatus').value = '';
    document.getElementById('filterStartDate').value = '';
    document.getElementById('filterEndDate').value = '';
    document.getElementById('filterIncludeArchived').checked = false;
    
    if (currentUser.role === 'Admin') {
        const leaderFilter = document.getElementById('filterLeader');
        if (leaderFilter) leaderFilter.value = '';
    }
    
    loadFlags();
}

// Audit Functions
async function loadAuditLogs() {
    if (currentUser.role !== 'Admin') return;
    
    try {
        const response = await fetch(`${API_BASE}/audit`, {
            headers: {
                'Authorization': `Bearer ${authToken}`
            }
        });
        
        if (response.ok) {
            const logs = await response.json();
            updateAuditTable(logs);
        } else {
            throw new Error('Erro ao carregar logs de auditoria');
        }
    } catch (error) {
        console.error('Erro ao carregar logs de auditoria:', error);
        showError('Erro ao carregar logs de auditoria');
    }
}

function updateAuditTable(logs) {
    const tbody = document.getElementById('auditTableBody');
    
    if (logs.length === 0) {
        tbody.innerHTML = '<tr><td colspan="4" class="text-center py-4 text-gray-500">Nenhum log encontrado</td></tr>';
        return;
    }
    
    tbody.innerHTML = logs.map(log => `
        <tr class="hover:bg-gray-50 dark:hover:bg-gray-700">
            <td class="px-6 py-4 whitespace-nowrap text-sm text-gray-900 dark:text-white">${formatDate(log.timestamp)}</td>
            <td class="px-6 py-4 whitespace-nowrap text-sm text-gray-900 dark:text-white">${log.username || 'Sistema'}</td>
            <td class="px-6 py-4 whitespace-nowrap text-sm text-gray-900 dark:text-white">${log.action}</td>
            <td class="px-6 py-4 text-sm text-gray-900 dark:text-white">${log.details}</td>
        </tr>
    `).join('');
}

// Utility Functions
function formatDate(dateString) {
    const date = new Date(dateString);
    return date.toLocaleDateString('pt-BR') + ' ' + date.toLocaleTimeString('pt-BR', { hour: '2-digit', minute: '2-digit' });
}

function getGravityColor(gravity) {
    switch(gravity) {
        case 'Amarela': return 'bg-yellow-100 text-yellow-800';
        case 'Laranja': return 'bg-orange-100 text-orange-800';
        case 'Vermelha': return 'bg-red-100 text-red-800';
        default: return 'bg-gray-100 text-gray-800';
    }
}

function getStatusColor(status) {
    switch(status) {
        case 'Aberta': return 'bg-red-100 text-red-800';
        case 'Em análise': return 'bg-yellow-100 text-yellow-800';
        case 'Em plano de ação': return 'bg-blue-100 text-blue-800';
        case 'Concluída': return 'bg-green-100 text-green-800';
        case 'Arquivada': return 'bg-gray-100 text-gray-800';
        default: return 'bg-gray-100 text-gray-800';
    }
}

